unit uPlugin;
interface
uses Windows, DVSTUtils, uEditor, DAEffect, DAEffectX, DAudioEffect, DAudioEffectX;

const
 p4=1.0e-24;
 kID = 'TBDM'; // unique plugin identifier, has to be different for every plugin
 kChannelID='TBDM'; // string displayed in the VSTi channel mixer
 kEffectName = 'Decimator'; // effect name
 kProduct = 'Decimator'; // product name
 kVendor = 'Tobybear'; // vendor name
 kIsSynth = false; //false=audio effect, true=synth
 kNumInputs = 2; // number of inputs
 kNumOutputs = 2; // number of outputs
 kCanMono = true; // can be fed with mono signals?
 kCanReplacing = true; //processreplacing() is called instead of process()

 kWet=0;
 kbits=1;
 kshrate=2;
 kCutoff=3;
 kResonance=4;
 kFType=5;
 kOutVol=6;

 kNumParams = 7;
 kNumPrograms = 8;

 type APluginEditor=class;
 APluginProgram=class
 private
  fWet,fbits,fshrate,fCutoff,fResonance,fFType,fOutVol:single;
  name:array[0..50] of char;
 public
  constructor Create;
 end;

 APlugin=class(AudioEffectX)
 private
  m:longint;
  cnt,yout1,yout2:single;
  old1,old2,old1a,old2a:single;

  fWet,fbits,fshrate,fCutoff,fResonance,fFType,fOutVol:single;
  done:boolean;
 public
  programs:array[0..kNumPrograms-1] of APluginProgram;
  vu:single;
  function GetPluginEditor:APluginEditor;
  constructor CreateAPlugin(audioMaster:TAudioMasterCallbackFunc);virtual;
  destructor Destroy;override;
  procedure DoProcess(var i1:single;var i2:single);
  procedure process(inputs,outputs:PPSingle;sampleframes:Longint);override;
  procedure processReplacing(inputs,outputs:PPSingle;sampleframes:Longint);override;
  procedure setProgram(aProgram:Longint);override;
  function canDo(text:pchar):longint;override;
  procedure setProgramName(name:PChar);override;
  procedure getProgramName(name:PChar);override;
  procedure setParameter(index:Longint;value:Single);override;
  function getParameter(index:Longint):Single;override;
  function getVu:Single;override;
  procedure DoFilter(var i1,i2:single;cutoff,res:single);
  function processEvents(ev:PVSTEvents):longint;override;
  procedure resume;override;
  procedure suspend;override;
  procedure getParameterLabel(index:longint;text:pchar);override;
  procedure getParameterDisplay(index:longint;text:pchar);override;
  procedure getParameterName(index:longint;text:pchar);override;
  function getOutputProperties(index:longint;properties:PVstPinProperties):boolean;override;
  function getProgramNameIndexed(category,index:longint;text:pchar):boolean;override;
  function getEffectName(name:pchar):boolean;override;
  function getVendorString(text:pchar):boolean;override;
  function getProductString(text:pchar):boolean;override;
  function getVendorVersion:longint;override;
  property PluginEditor:APluginEditor read GetPluginEditor;
 end;

 APluginEditor=class(AEffEditor)
 private
  r:ERect;
  useCount:Longint;
  Editor:TPluginEditorWindow;
  systemWindow:HWnd;
  function GetPlugin:APlugin;
 public
  constructor Create(effect:AudioEffect);override;
  destructor Destroy;override;
  function getRect(var rect:PERect):Longint;override;
  function open(ptr:Pointer):Longint;override;
  procedure close;override;
  procedure idle;override;
  procedure update;override;
  property Plugin:APlugin read GetPlugin;
 end;

implementation
uses SysUtils;

function APlugin.canDo(text:pchar):longint;
begin
 Result:=-1;
 if StrComp(text, 'receiveVstEvents') = 0 then Result := 1;
 if StrComp(text, 'receiveVstMidiEvent') = 0 then Result := 1;
end;

constructor APluginProgram.Create;
begin
 inherited Create;
 fWet:=1;
 fbits:=0.5;
 fshrate:=0.5;
 fCutoff:=1;
 fResonance:=0;
 fFType:=0;
 fOutVol:=1;
 StrCopy(name,'Init'); // set program name
end;

constructor APlugin.CreateAPlugin(audioMaster:TAudioMasterCallbackFunc);
var i:integer;
begin
 inherited Create(audioMaster,knumprograms,kNumParams);
 for i:=kNumPrograms-1 downto 0 do programs[i] := APluginProgram.Create;

 editor:=APluginEditor.Create(Self);
 randomize;
 suspend;
 old1:=0;
 old2:=0;
 old1a:=0;
 old2a:=0;
 cnt:=1;

 hasVu(false);
 setNumInputs(KNumInputs);
 setNumOutputs(KNumOutputs);
 canMono(KCanMono);
 canProcessReplacing(KCanReplacing);
 isSynth(KIsSynth);
 setUniqueID(FourCharToLong(kID[1], kID[2], kID[3], kID[4]));

 StrCopy(programs[0].name, 'Init Preset 01');
 StrCopy(programs[1].name, 'Init Preset 02');
 StrCopy(programs[2].name, 'Init Preset 03');
 StrCopy(programs[3].name, 'Init Preset 04');
 StrCopy(programs[4].name, 'Init Preset 05');
 StrCopy(programs[5].name, 'Init Preset 06');
 StrCopy(programs[6].name, 'Init Preset 07');
 StrCopy(programs[7].name, 'Init Preset 08');
 curprogram:=0;
 setProgram(0);
end;

destructor APlugin.Destroy;
var i:integer;
begin
 inherited Destroy;

 // destroy the created programs
 for i:=0 to kNumPrograms-1 do
 begin
  programs[i].Free;
  programs[i]:=nil;
 end;
end;

procedure APlugin.setProgram(aProgram: Longint);
begin
 if (aprogram<0)or(aprogram>knumprograms-1) then exit;

 curProgram := aProgram;
 SetParameter(kWet,programs[curprogram].fWet);
 SetParameter(kbits,programs[curprogram].fbits);
 SetParameter(kshrate,programs[curprogram].fshrate);
 SetParameter(kCutoff,programs[curprogram].fCutoff);
 SetParameter(kResonance,programs[curprogram].fResonance);
 SetParameter(kFType,programs[curprogram].fFType);
 SetParameter(kOutVol,programs[curprogram].fOutVol);
 if done then exit;

 if assigned(editor) then editor.update;
end;

procedure APlugin.setProgramName(name:PChar);
begin
 StrCopy(programs[curProgram].name, name);
 if done then exit;
 if assigned(editor) then editor.update;
end;

procedure APlugin.getProgramName(name: PChar);
begin
 StrCopy(name, programs[curProgram].name);
end;

procedure APlugin.suspend;
begin
end;

function APlugin.getVu:Single;
var cvu:Single;
begin
 cvu:=vu;
 vu:=0;
 Result:=cvu;
end;

procedure APlugin.setParameter(index:Longint;value:Single);
begin
 if (value>1) then value:=1 else if (value<0) then value:=0;

 case index of
  kWet:begin fWet:=value;programs[curprogram].fWet:=value end;
  kbits:begin
   m:=round(value*15)+1;
   m:=1 shl (m-1);
   cnt:=1;
   fbits:=value;programs[curprogram].fbits:=value
  end;
  kshrate:begin fshrate:=value;programs[curprogram].fshrate:=value end;
  kCutoff:begin
   fCutoff:=value;programs[curprogram].fCutoff:=value;
  end;
  kResonance:begin
   fResonance:=value;programs[curprogram].fResonance:=value;
  end;
  kFType:begin fFType:=value;programs[curprogram].fFType:=value end;
  kOutVol:begin fOutVol:=value;programs[curprogram].fOutVol:=value end;
 end;
 if assigned(editor) then editor.update;
end;

function APlugin.getParameter(index: Longint): Single;
var j:single;
begin
 case index of
  kWet:j:=fWet;
  kbits:j:=fbits;
  kshrate:j:=fshrate;
  kCutoff:j:=fCutoff;
  kResonance:j:=fResonance;
  kFType:j:=fFType;
  kOutVol:j:=fOutVol;
 else
  j:=0;
 end;
 Result:=j;
end;

procedure APlugin.DoFilter(var i1,i2:single;cutoff,res:single);
var fb:single;
begin
 if cutoff>0.9 then Cutoff:=0.9 else if cutoff<0.1 then Cutoff:=0.1;
 if res>0.8 then res:=0.8;
 fb:=res+res/(1-cutoff);
 old1:=old1+cutoff*(i1-old1+fb*(old1-old2))+p4;
 old2:=old2+cutoff*(old1-old2)+p4;
 old1a:=old1a+cutoff*(i2-old1a+fb*(old1a-old2a))+p4;
 old2a:=old2a+cutoff*(old1a-old2a)+p4;

 //limit coeffcients - not very elegant, but else filter
 //is particularly unstable with high resonance and low
 //sample&hold rates
 if old1>1 then old1:=1 else if old1<-1 then old1:=-1;
 if old2>1 then old2:=1 else if old2<-1 then old2:=-1;
 if old1a>1 then old1a:=1 else if old1a<-1 then old1a:=-1;
 if old2a>1 then old2a:=1 else if old2a<-1 then old2a:=-1;

 if fFType<0.5 then
 begin
  i1:=old2;
  i2:=old2a;
 end
 else begin
  i1:=i1-old2;
  i2:=i2-old2a;
 end;
end;

procedure APlugin.DoProcess(var i1:single;var i2:single);
begin
 cnt:=cnt+fshrate;
 if (cnt>1) then
 begin
  cnt:=cnt-1;
  yout1:=round(i1*m)/m;
  yout2:=round(i2*m)/m;
 end;

 DoFilter(yout1,yout2,fcutoff,fresonance);

 i1:=fOutVol*((1-fWet)*i1+fWet*yout1);
 i2:=fOutVol*((1-fWet)*i2+fWet*yout2);

 if i1>1 then i1:=1 else if i1<-1 then i1:=-1; 
 if i2>1 then i2:=1 else if i2<-1 then i2:=-1;
end;


procedure APlugin.process(inputs,outputs:PPSingle;sampleframes:Longint);
var In1,In2,Out1,Out2:PSingle;
    i1,i2:Single;
    i:Integer;
begin
 In1:=inputs^;
 In2:=PPSingle(Longint(inputs)+SizeOf(PSingle))^;
 Out1:=outputs^;
 Out2:=PPSingle(Longint(outputs)+SizeOf(PSingle))^;
 for i:=0 to sampleFrames-1 do
 begin
  i1:=In1^;
  i2:=In2^;
  DoProcess(i1,i2);
  Out1^:=Out1^+i1;
  Out2^:=Out2^+i2;
  Inc(In1);
  Inc(In2);
  Inc(Out1);
  Inc(Out2);
 end;
end;

procedure APlugin.processReplacing(inputs, outputs: PPSingle; sampleframes: Longint);
var In1,In2,Out1,Out2:PSingle;
    i1,i2:Single;
    i:Integer;
begin
 In1:=inputs^;
 In2:=PPSingle(Longint(inputs)+SizeOf(PSingle))^;
 Out1:=outputs^;
 Out2:=PPSingle(Longint(outputs)+SizeOf(PSingle))^;
 for i:=0 to sampleFrames-1 do
 begin
  i1:=In1^;
  i2:=In2^;
  DoProcess(i1,i2);
  Out1^:=i1;
  Out2^:=i2;
  Inc(In1);
  Inc(In2);
  Inc(Out1);
  Inc(Out2);
 end;
end;

function APlugin.GetPluginEditor:APluginEditor;
begin
 Result:=(editor as APluginEditor);
end;

constructor APluginEditor.Create(effect:AudioEffect);
begin
 inherited Create(effect);
 useCount:=0;
end;

destructor APluginEditor.Destroy;
begin
 if assigned(editor) then
 begin
  Plugin.done:=true;
  Editor.Free;
  Editor:=nil;
  systemWindow:=0;
 end;
 inherited Destroy;
end;

function APluginEditor.getRect(var rect:PERect):Longint;
begin
 r.top:=0;
 r.left:=0;
 r.bottom:=231;
 r.right:=244;
 rect:=@r;
 Result := 1;
end;

function APluginEditor.open(ptr:Pointer):Longint;
begin
 systemWindow := HWnd(ptr);
 Inc(useCount);
 if (useCount=1)or (not assigned(editor)) then
 begin
  Editor:=TPluginEditorWindow.CreateParented(systemWindow);
  Editor.SetBounds(0,0,Editor.Width,Editor.Height);
  Editor.fWet:=Plugin.fWet;
  Editor.fbits:=Plugin.fbits;
  Editor.fshrate:=Plugin.fshrate;
  Editor.fCutoff:=Plugin.fCutoff;
  Editor.fResonance:=Plugin.fResonance;
  Editor.fFType:=Plugin.fFType;
  Editor.fOutVol:=Plugin.fOutVol;
  Editor.Effect:=Self.Plugin;
 end;
 Plugin.done:=false;
 Editor.Update;
 Editor.Show;
 editor.UpdaterTimer(nil);
 Result:=1;
end;

procedure APluginEditor.close;
begin
 if assigned(editor) then editor.visible:=false;
 Dec(useCount);
 if usecount<0 then usecount:=0;
 if useCount=0 then
 begin
  Plugin.done:=true;
  Editor.Free;
  Editor:=nil;
  systemWindow:=0;
 end;
end;

procedure APluginEditor.idle;
begin
 if (Plugin.done) or (not Assigned(Editor)) then exit;
end;

procedure APluginEditor.update;
begin
 if (Plugin.done) or (not Assigned(Editor)) then exit;
 Editor.fWet:=Plugin.fWet;
 Editor.fbits:=Plugin.fbits;
 Editor.fshrate:=Plugin.fshrate;
 Editor.fCutoff:=Plugin.fCutoff;
 Editor.fResonance:=Plugin.fResonance;
 Editor.fFType:=Plugin.fFType;
 Editor.fOutVol:=Plugin.fOutVol;
end;

function APluginEditor.GetPlugin: APlugin;
begin
 Result:=(effect as APlugin);
end;

function APlugin.getEffectName(name:pchar): boolean;
begin
 StrCopy(name,kEffectName);
 Result:=TRUE;
end;

function APlugin.getVendorString(text:pchar):boolean;
begin
 StrCopy(text, kVendor);
 Result:=TRUE;
end;

function APlugin.getProductString(text:pchar):boolean;
begin
 StrCopy(text,kProduct);
 Result:=TRUE;
end;

function APlugin.getVendorVersion:longint;
begin
 Result:=1; // return version number
end;

function APlugin.getOutputProperties(index:longint;properties:PVstPinProperties):boolean;
begin
 Result:=false;
 if (index<kNumOutputs) then
 begin
  StrCopy(properties^.vLabel, pchar(Format(kChannelID+' %d', [index+1])));
  properties^.flags:=kVstPinIsActive;
  if (index<2) then
   properties^.flags:=properties^.flags or kVstPinIsStereo;
  Result:=true;
 end;
end;

function APlugin.getProgramNameIndexed(category,index:longint;text:pchar):boolean;
begin
 Result:=false;
 if (index<kNumPrograms) then
 begin
  StrCopy(text,programs[index].name);
  Result:=true;
 end;
end;

procedure APlugin.getParameterName(index:longint;text:pchar);
begin
 case index of
  kWet:StrCopy(text, 'wet/dry mix');
  kbits:StrCopy(text, 'bits');
  kshrate:StrCopy(text, 'sh rate');
  kCutoff:StrCopy(text, 'filter-cutoff');
  kResonance:StrCopy(text, 'filter-reso');
  kfType:StrCopy(text, 'filter-type');
  kOutVol:StrCopy(text, 'output vol');
  else StrCopy(text, 'reserved');
  end;
end;

procedure APlugin.getParameterDisplay(index:longint;text:pchar);
begin
 case index of
  kWet:float2string(fWet, text);
  kbits:float2string(fbits, text);
  kshrate:float2string(fshrate, text);
  kCutoff:float2string(fCutoff, text);
  kResonance:float2string(fResonance, text);
  kFType:float2string(fFType, text);
  kOutVol:float2string(fOutVol, text);
  else float2string(0, text);
 end;
end;

procedure APlugin.getParameterLabel(index:longint;text:pchar);
begin
 StrCopy(text, '%');
end;

function APlugin.processEvents(ev:PVstEvents):longint;
var note,k,i,status:longint;
    nvol:single;
    event:PVstMidiEvent;
    midiData:array[0..3] of byte;
begin
 for i:=0 to ev^.numEvents-1 do
 if (ev.events[i].vtype=kVstMidiType) then
 begin
  event:=PVstMidiEvent(ev^.events[i]);
  for k:=0 to 3 do midiData[k]:=event.midiData[k];
  status:=midiData[0] AND $f0; // channel information is removed

  if (status=$B0) then // midi CC ?
  begin
   note:=event^.mididata[1]; // midi CC#
   nvol:=event^.mididata[2]/127; // CC data
   case note of
   70:fshrate:=nvol;
   71:fbits:=nvol;
   72:fcutoff:=nvol;
   73:fresonance:=nvol;
   74:fwet:=nvol;
   75:foutvol:=nvol;
   else
   end;
   if assigned(editor) then editor.update;
  end;
 end;
 
 Result:=1;
end;

procedure APlugin.resume;
begin
 wantEvents(1);
end;

end.

